//////////////////////////////////////////////////////////////////////
// kernel_prx.c - ELF Loader thru a module
//////////////////////////////////////////////////////////////////////

#include <pspkernel.h>
#include <pspiofilemgr.h>
#include <pspsdk.h>
#include <string.h>
#include <stdio.h>
#include <pspdebug.h>
#include "regex.h"

#define MODULE_NAME			"PBPLOADER"
#define VERSION 			"1"
#define REBOOT_TIME 		3000000
#define printf				pspDebugScreenPrintf

#define	NUM_FIELDS			3
#define ARG_SEPARATOR 		";"

PSP_MODULE_INFO(MODULE_NAME, 0x1000, 1, 1);
PSP_MAIN_THREAD_ATTR(0);

static char * rebootPath;


//////////////////////////////////////////////////////////////////////
// Prototypes
//////////////////////////////////////////////////////////////////////
int unload_loader(char * moduleName);
void rebootApp(void);
int run_elf(char * elfPath);
void crash_handler(PspDebugRegBlock *regs);
int unload_loader(char * moduleName);


//////////////////////////////////////////////////////////////////////
// Reboot entry point ( Hooked to sceKernelExitGame() )
//////////////////////////////////////////////////////////////////////
void rebootApp(void)
{
	struct SceKernelLoadExecParam execParam;

	execParam.size = sizeof(execParam);
	execParam.argp = rebootPath;
	execParam.args = strlen(rebootPath);
	execParam.key = NULL;

	sceKernelLoadExec(rebootPath, &execParam);
}

//////////////////////////////////////////////////////////////////////
// Load module
//////////////////////////////////////////////////////////////////////
SceUID load_module(const char *path, int flags, int type)
{
	SceKernelLMOption option;
	SceUID mpid;

	/* If the type is 0, then load the module in the kernel partition, otherwise load it
	   in the user partition. */
	if (type == 0)
	{
		mpid = 1;
	}
	else
	{
		mpid = 2;
	}

	memset(&option, 0, sizeof(option));
	option.size = sizeof(option);
	option.mpidtext = mpid;
	option.mpiddata = mpid;
	option.position = 0;
	option.access = 1;

	return sceKernelLoadModule(path, flags, type > 0 ? &option : NULL);
}

//////////////////////////////////////////////////////////////////////
// Thread
//////////////////////////////////////////////////////////////////////
int main_thread(SceSize args, void *argp)
{
	char * fields[NUM_FIELDS];
	int nbArgs;

	pspDebugScreenInit();
	//pspDebugInstallKprintfHandler(NULL);

	pspSdkInstallNoDeviceCheckPatch();
	pspSdkInstallNoPlainModuleCheckPatch();
	pspDebugInstallErrorHandler(crash_handler);

	// Patch the Kernel@sceGameExit
	#define J(addr)       (0x08000000 | (0xFFFFFFF & (((u32)(addr)) >> 2)))
	switch (sceKernelDevkitVersion())
	{
		case 0x01050001: // 1.5
			*((unsigned int *)0x8806882c) = J(rebootApp);
			*((unsigned int *)0x88068830) = 0x00000000; // nop
			break;
		case 0x01000300: // 1.0
			*((unsigned int *)0x88064118) = J(rebootApp);
			*((unsigned int *)0x8806411c) = 0x00000000; // nop
			break;
	}

  	nbArgs = split((char *) argp , fields, NUM_FIELDS, ARG_SEPARATOR);


	// Wrong number of args
	if( nbArgs != NUM_FIELDS)
	{
		crash_handler(0);
	}

	rebootPath = fields[2];

	int res_unload = unload_loader(fields[0]);	// Unload the prx loader (optional)

	if(res_unload != -1)
	{
		int res_run = run_elf(fields[1]);
		if(res_run == -1)
		{
			crash_handler(0);
		}
	}
	else
	{
		crash_handler(0);
	}

	pspDebugScreenClear();
	sceKernelExitDeleteThread(0);
	return 0;
}

int module_start(SceSize args, void *argp)
{
	int thid;
	u32 func;

	// Relocate main thread below normal 0x89000000 to avoid data crunching
	func = (u32) main_thread;
	func |= 0x80000000;

	/* Create a high priority thread */
	thid = sceKernelCreateThread("main_thread", (void *) func, 0x20, 0x10000, 0, NULL);
	if(thid >= 0)
	{
		sceKernelStartThread(thid, args, argp);
	}

	return 0;
}

int module_stop(SceSize args, void *argp)
{
	return 0;
}

int run_elf(char * elfPath)
{
	SceUID modid_loaded, modid_started;

	// Load module
	modid_loaded = load_module( elfPath, 0, 0);

	// Start module with input path
	if(modid_loaded >= 0)
	{
		int ret_val = 1;

		modid_started = sceKernelStartModule(modid_loaded, strlen(elfPath) + 1, elfPath, &ret_val, NULL);

		if(modid_started < 0)
		{
			printf("Error Starting Application %08X\n", modid_started);
			return -1;
		}
	}
	else
	{
		printf("Error Loading Application %08X\n", modid_loaded);
		return -1;
	}

	return 0;
}

int unload_loader(char * moduleName)
{
	SceModule *mod;
	SceUID modid;
	int ret_stop, ret_unload = 0;
	int status;

	mod = sceKernelFindModuleByName(moduleName);
	if(mod != NULL)
	{
		/* Stop module */
		modid = mod->modid;

		ret_stop = sceKernelStopModule(modid, 0, NULL, &status, NULL);
		if(ret_stop >= 0)
		{
			ret_unload = sceKernelUnloadModule(modid);
			if(ret_unload < 0)
			{
				return -1;
			}
		}
		else
		{
			return -1;
		}
	}
	else
	{
		return -1;
	}

	return 0;
}

void crash_handler(PspDebugRegBlock *regs)
{
	sceKernelDelayThread(REBOOT_TIME);
	rebootApp();
}

